import { useState, useEffect } from 'react';
import { Link, useLocation } from 'wouter';
import { Button } from "@/components/ui/button";

export function Header() {
  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);
  const [isScrolled, setIsScrolled] = useState(false);
  const [location] = useLocation();

  const toggleMobileMenu = () => {
    setIsMobileMenuOpen(!isMobileMenuOpen);
  };

  useEffect(() => {
    const handleScroll = () => {
      setIsScrolled(window.scrollY > 10);
    };
    
    window.addEventListener('scroll', handleScroll);
    
    return () => {
      window.removeEventListener('scroll', handleScroll);
    };
  }, []);

  const isActive = (path: string) => {
    return location === path;
  };

  const navLinks = [
    { label: 'Home', path: '/' },
    { label: 'Jira', path: '/jira' },
    { label: 'GLPI', path: '/glpi' },
    { label: 'ESMP', path: '/esmp' },
    { label: 'About', path: '/#about' },
    { label: 'Contact', path: '/#contact' },
  ];

  return (
    <header className={`bg-white shadow-sm fixed w-full z-50 transition-all ${isScrolled ? 'py-2' : 'py-4'}`}>
      <nav className="container mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex justify-between items-center">
          <div className="flex items-center">
            <Link href="/" className="flex items-center gap-2">
              <div className="w-10 h-10 bg-primary-600 text-white rounded-lg flex items-center justify-center">
                <i className="fas fa-headset text-xl"></i>
              </div>
              <span className="text-xl font-bold text-slate-900">
                ServiceDesk<span className="text-primary-600">Experts</span>
              </span>
            </Link>
          </div>
          
          {/* Desktop Navigation */}
          <div className="hidden md:flex items-center space-x-8">
            {navLinks.map((link, index) => (
              <Link
                key={index}
                href={link.path}
                className={`font-medium ${
                  isActive(link.path)
                    ? 'text-primary-600'
                    : 'text-slate-700 hover:text-primary-600 transition-colors'
                }`}
              >
                {link.label}
              </Link>
            ))}
            <Button asChild className="ml-4">
              <Link href="/#contact">Get Started</Link>
            </Button>
          </div>
          
          {/* Mobile Menu Button */}
          <button 
            type="button" 
            className="md:hidden text-slate-800" 
            onClick={toggleMobileMenu}
            aria-expanded={isMobileMenuOpen}
            aria-label="Toggle navigation menu"
          >
            <i className="fas fa-bars text-2xl"></i>
          </button>
        </div>
        
        {/* Mobile Navigation */}
        <div className={`md:hidden ${isMobileMenuOpen ? 'block' : 'hidden'}`}>
          <div className="pt-2 pb-4 space-y-1">
            {navLinks.map((link, index) => (
              <Link
                key={index}
                href={link.path}
                className={`block px-3 py-2 font-medium rounded-md ${
                  isActive(link.path)
                    ? 'bg-primary-50 text-primary-600'
                    : 'text-slate-700 hover:bg-primary-50 hover:text-primary-600'
                }`}
                onClick={() => setIsMobileMenuOpen(false)}
              >
                {link.label}
              </Link>
            ))}
            <Link
              href="/#contact"
              className="block px-3 py-2 mt-2 bg-primary-600 text-white rounded-lg font-medium hover:bg-primary-700 text-center"
              onClick={() => setIsMobileMenuOpen(false)}
            >
              Get Started
            </Link>
          </div>
        </div>
      </nav>
    </header>
  );
}
