import { db } from "./index";
import * as schema from "@shared/schema";
import { eq } from "drizzle-orm";

async function seed() {
  try {
    console.log("Starting database seeding...");
    
    // Check if services already exist
    const existingServices = await db.query.services.findMany();
    
    if (existingServices.length === 0) {
      console.log("Seeding services...");
      
      // Insert services
      const jiraService = await db.insert(schema.services)
        .values({
          name: "Jira Service Management",
          slug: "jira",
          shortDescription: "Optimize your Jira Service Management instance with custom workflows, automation, and integrations.",
          iconClass: "fab fa-jira",
          colorClass: "text-primary-600",
          borderClass: "bg-primary-600",
        })
        .returning();
      
      const glpiService = await db.insert(schema.services)
        .values({
          name: "GLPI Configuration",
          slug: "glpi",
          shortDescription: "Implement and customize GLPI to create a comprehensive IT asset and service management system.",
          iconClass: "fas fa-ticket-alt",
          colorClass: "text-secondary-600",
          borderClass: "bg-secondary-600",
        })
        .returning();
      
      const esmpService = await db.insert(schema.services)
        .values({
          name: "ESMP Implementation",
          slug: "esmp",
          shortDescription: "Configure Enterprise Service Management Platform to streamline cross-department service delivery.",
          iconClass: "fas fa-server",
          colorClass: "text-accent-500",
          borderClass: "bg-accent-500",
        })
        .returning();
      
      // Insert service features
      const jiraFeatures = [
        "Custom workflow configuration", 
        "Automation rule setup", 
        "SLA & reporting customization", 
        "Third-party app integration"
      ];
      
      const glpiFeatures = [
        "Asset management setup", 
        "Helpdesk configuration", 
        "Plugin installation & setup", 
        "Custom dashboard creation"
      ];
      
      const esmpFeatures = [
        "Multi-department setup", 
        "Process automation", 
        "Self-service portal configuration", 
        "Integration with existing systems"
      ];
      
      for (const feature of jiraFeatures) {
        await db.insert(schema.serviceFeatures)
          .values({
            serviceId: jiraService[0].id,
            feature
          });
      }
      
      for (const feature of glpiFeatures) {
        await db.insert(schema.serviceFeatures)
          .values({
            serviceId: glpiService[0].id,
            feature
          });
      }
      
      for (const feature of esmpFeatures) {
        await db.insert(schema.serviceFeatures)
          .values({
            serviceId: esmpService[0].id,
            feature
          });
      }
      
      console.log("Services seeded successfully!");
    } else {
      console.log("Services already exist, skipping service seeding.");
    }
    
    // Check if testimonials already exist
    const existingTestimonials = await db.query.testimonials.findMany();
    
    if (existingTestimonials.length === 0) {
      console.log("Seeding testimonials...");
      
      // Insert testimonials
      await db.insert(schema.testimonials)
        .values([
          {
            name: "John Davis",
            position: "IT Director",
            company: "Global Finance Corp",
            rating: 5,
            testimonial: "The team's expertise in Jira Service Management completely transformed our IT helpdesk. The custom workflows they implemented have reduced our resolution times by over 40%.",
            initials: "JD",
            bgColor: "bg-primary-100",
            textColor: "text-primary-600",
          },
          {
            name: "Sarah Mitchell",
            position: "CIO",
            company: "Healthcare Solutions",
            rating: 5,
            testimonial: "Their GLPI configuration services helped us implement a comprehensive asset management system that integrates perfectly with our service desk. The customization was exactly what we needed.",
            initials: "SM",
            bgColor: "bg-secondary-100",
            textColor: "text-secondary-600",
          },
          {
            name: "Robert Kim",
            position: "Operations Manager",
            company: "Tech Innovations",
            rating: 4,
            testimonial: "The ESMP implementation was seamless, and the team's attention to detail was impressive. They understood our cross-departmental needs and configured the platform accordingly.",
            initials: "RK",
            bgColor: "bg-accent-100",
            textColor: "text-accent-600",
          }
        ]);
      
      console.log("Testimonials seeded successfully!");
    } else {
      console.log("Testimonials already exist, skipping testimonial seeding.");
    }
    
    console.log("Database seeding completed successfully!");
  } catch (error) {
    console.error("Error seeding database:", error);
  }
}

seed();
